/************************************************************************
 * @file: AudioFactory.h
 *
 * @version: 0.1
 *
 * @description: This header file contains definition for class Factory.
 * Factory class will be used by application to select the Backend Libraries,
 * based on the Backend Name [eg: Alsa] provided by application for
 * Factory class. This class provides handle for the selected Backend to
 * Application.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Thouseef Ahamed, tahamed@de.adit-jv.com 2015
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef _ADIT_UTILITY_AUDIOFACTORY_H_
#define _ADIT_UTILITY_AUDIOFACTORY_H_

#include <vector>
#include <string>
#include <map>
#include <memory>

#include "AudioStreaming.h"
#include "AudioBackend.h"

namespace adit
{

namespace utility
{

namespace audio
{

class Factory
{

public:

    ~Factory(void);

    /**
     * The function loads Factory instance and uses the given Logging handle
     * @param[IN] loggingHandle Handle used for Logging
     */
    static Factory* Instance(LoggingInterface& loggingHandle);

    /**
     * The function loads Factory instance and uses stdout Logging
     */
    static Factory* Instance();

    /**
     * A function that looks for supported audio backend in specific directory and creates list of supported audio backends.
     * \note
     * The specific directory (BE_LIBRARY_PATH) for audio Backends is mentioned in AudioFactory.cpp.
     * @param[OUT] backends The backends list.
     * @returns OK      if the specific directory is able to open and list is created.
     * @returns FAILURE if the specific directory is not able to open.
     */
    AudioError getSupportedBe(std::vector<std::string>& backends);

    /**
     * The function loads the backend and return the handle for selected backend.
     * @param[IN|OUT] Backend library name which shall be selected.
     *                It can be empty and the first library found in system will be selected.
     * @param[IN]     streamingHandle Handle of Streaming class used for Logging and data processing
     * @returns shared_ptr<Backend> handle providing the interface.
     * @returns NULL if the Backend is not available or the audio library is not able to open.
     *
     * @note If the shared pointer goes out of scope, Backend handle will be deleted.
    */
    std::shared_ptr<Backend> createBackend(std::string& backend, Streaming& streamingHandle);

private:

    /**
     * This is a Singleton and therefore in private section
     * @param[IN] loggingHandle Handle used for Logging.
     */
    Factory(LoggingInterface& loggingHandle);

    LoggingInterface&         mLoggingHandle;
    static Factory*           singleton;

};

} /* namespace audio */

} /* namespace utility */

} /* namespace adit */

#endif /* _ADIT_UTILITY_AUDIOFACTORY_H_ */
